/* -*- Mode: C; tab-width: 8 -*- */

#include "Python.h"

#include <sys/poll.h>

static PyObject * PollError;

static
PyObject *
Py_Poll (PyObject * self, PyObject * args)
{
  PyObject * fd_list;
  int timeout = 0;
  
  if (!PyArg_ParseTuple (args, "O!|i", &PyList_Type, &fd_list, &timeout)) {
    return NULL;
  } else {
    int nfds = PyList_Size (fd_list);
    struct pollfd * fd_array = PyMem_NEW (struct pollfd, nfds);
    int poll_result;

    /* Create and initialize the array of pollfd structs */

    /* fprintf (stderr, "nfds == %d\n", nfds); */

    if (!fd_array) {
      PyErr_SetString (PyExc_MemoryError, "could not allocate memory for poll array");
      return NULL;
    } else {
      int i;
      for (i=0; i < nfds; i++) {
	PyObject * triple = PyList_GetItem (fd_list, i);
	if (!PyArg_ParseTuple (triple, "ii", &(fd_array[i].fd), &(fd_array[i].events))) {
	  PyMem_DEL (fd_array);
	  PyErr_SetString (
              PyExc_ValueError,
	      "each element of poll list must be a tuple of (<descriptor>, <events>), both integers"
	      );
	  return NULL;
	}
	fd_array[i].revents = 0;
      }
    }

    /* call poll() */
    
    Py_BEGIN_ALLOW_THREADS
    poll_result = poll (fd_array, nfds, timeout);
    Py_END_ALLOW_THREADS

    /* fprintf (stderr, "poll_result() => %d\n", poll_result); */

    if (poll_result < 0) {
      PyMem_DEL (fd_array);
      PyErr_SetFromErrno (PollError);
      return NULL;
    } else {
      
      /* build the result list */

      PyObject * result_list = PyList_New (poll_result);
      if (!result_list) {
	PyMem_DEL (fd_array);
	return NULL;
      } else {
	int i = 0;
	int j = 0;
	for (j=0; j < poll_result; j++) {
	  /* skip to the next fired descriptor */
	  while (!fd_array[i].revents) {
	    i++;
	  }
	  PyList_SetItem (result_list, j, Py_BuildValue ("(ii)", fd_array[i].fd, fd_array[i].revents));
	  i++;
	}
      }
      PyMem_DEL (fd_array);
      return result_list;
    }
  }
}


static PyMethodDef poll_methods[] = {
  {"poll", Py_Poll, 1},
  {0, 0},
};

/* Convenience routine to export an integer value.
 *
 * Errors are silently ignored, for better or for worse...
 */
static void
insint (PyObject * d, char * name, int value)
{
  PyObject *v = PyInt_FromLong((long) value);
  if (!v || PyDict_SetItemString(d, name, v))
    PyErr_Clear();
  Py_XDECREF(v);
}

void
initpoll (void)
{
  PyObject *m, *d;
  m = Py_InitModule ("poll", poll_methods);
  d = PyModule_GetDict (m);
  PollError = PyErr_NewException ("poll.error", NULL, NULL);
  PyDict_SetItemString (d, "error", PollError);

  insint (d, "POLLIN", POLLIN);
  insint (d, "POLLOUT", POLLOUT);
  insint (d, "POLLERR", POLLERR);

#ifdef POLLPRI
  insint (d, "POLLPRI", POLLPRI);
#endif
#ifdef POLLHUP
  insint (d, "POLLHUP", POLLHUP);
#endif
#ifdef POLLNVAL
  insint (d, "POLLNVAL", POLLNVAL);
#endif
#ifdef POLLRDNORM
  insint (d, "POLLRDNORM", POLLRDNORM);
#endif
#ifdef POLLRDBAND
  insint (d, "POLLRDBAND", POLLRDBAND);
#endif
#ifdef POLLWRNORM
  insint (d, "POLLWRNORM", POLLWRNORM);
#endif
#ifdef POLLWRBAND
  insint (d, "POLLWRBAND", POLLWRBAND);
#endif
#ifdef POLLMSG
  insint (d, "POLLMSG", POLLMSG);
#endif
  
}
